/* aterm.h
 *    ANSI/AVATAR/VT102 display module, version 1.0
 *    Programmed 1994, Feb 1995 by Mark D. Rejhon.
 *    EMail: marky@ottawa.com, ag115@freenet.carleton.ca
 *
 * WHAT IS IT?
 *    This is a high speed ANSI/AVATAR/VT102 printing module.  Supports
 *    AVATAR/0+, ANSI-BBS, and a subset of VT102, all simultaneously!
 *    This code even has a TTY mode as well, too :-)
 *    Blazing speed, translation table driven.  No inline assembly!
 *    Moderatly portable, no custom libraries required!
 *
 * FILES
 *    aterm.h  - The main include module for the display functions.
 *    adefs.h  - The definitions and constants module for aterm.h.
 *    avid_*.h - Platform/application-specific video display functions.
 *
 * LICENSE:
 *    This version can be freely modified and redistributed using the GNU
 *    public license.  Please refer to included "COPYING" file for details.
 *    Please submit ALL changes to this code via Internet Email to address
 *    ag115@freenet.carleton.ca  (permanent address, will forward Email to
 *    my current Internet Service Provider)
 *
 * INSTRUCTIONS:
 *    To include this file in your program, #include "aterm.c"
 *    The functions provided are 'ainit', 'aprint', 'afprint'
 *    Please refer to EASYANSI.C for an example of how to use these.
 */

/* Standard includes */
#include <stdio.h>
#include <stdarg.h>
#include <string.h>

/* Portable Definitions and constants */
#include "adefs.h"

#ifndef AVID_H
  /* Platform/Application specific video output functions.  Can be for any */
  /* platform, can be direct screen access, BIOS, termcap, ncurses, etc.   */
  #if 1
    /* Specific to TC++, screen video output. (Used by default) */
    #include "avid_tc.h"
  #elif 0
    /* Portable, Video output into internal buffer */
    #include "avid_mem.h"
  #elif 0
    /* Dummy functions, and instructions on how to customize to your needs */
    #include "avid_nul.h"
  #endif
#endif

/* ainit(vid, top, left, bot, right)
 *   PURPOSE
 *     Initializes the terminal engine. This should be called first before
 *     executing aprintf.
 *   REQUIRES
 *     vid   = pointer to video variables data structure.
 *     top   = absolute row of top border of window within video buffer.
 *     left  = absolute col of left border of window within video buffer.
 *     bot   = absolute row of bottom border of window within video buffer.
 *     right = absolute col of right border of window within video buffer.
 *   RESULT
 *     The engine is initialized and "aprintf" is ready to be called for
 *     this particular video buffer.
 */
void ainit(aterm_type *vid, int top, int left, int bot, int right)
{
  vid->buffer = NULL;
  vid->scrwidth = 80;
  vid->wtop = top;
  vid->wleft = left;
  vid->wbot = bot;
  vid->wright = right;
  vid->wtopzone = top;
  vid->wbotzone = bot;
  vid->height = (vid->wbot) - (vid->wtop) + 1;
  vid->width = (vid->wright) - (vid->wleft) + 1;
  vid->topzone = 1;
  vid->botzone = vid->height;
  vid->row = 1;
  vid->col = 1;
  vid->attr = 7;
  vid->insmode = FALSE;
  vid->saverow = top;
  vid->savecol = left;
  vid->codeptr = 0;
  vid->codelen = 0;
  vid->codetype = nul;
  vid->ansi = TRUE;
  vid->avatar = TRUE;
  vid->destbs = TRUE;
  vid->beep = TRUE;
}


/* aprint(vid, astring)
 *   PURPOSE
 *     This prints a variable length string with ANSI/AVT processing.
 *   REQUIRES
 *     vid     = pointer to video variables data structure.
 *     astring = string of data to be printed.
 *   RESULT
 *     The string is output to the video buffer with processing for codes
 *     including ANSI, AVATAR and a subset of VT102.  This function macro
 *     prints only characters leading up to (not including) the first null
 *     in the string.
 */
#define aprint(vid,astring) afprint(vid,astring,strlen(astring))


/* afprint(vid, astring, len)
 *   PURPOSE
 *     This prints a fixed length string with ANSI/AVT processing.
 *   REQUIRES
 *     vid     = pointer to video variables data structure.
 *     astring = string of data to be printed.
 *     len     = length of string.
 *   RESULT
 *     The string is output to the video buffer with processing for codes
 *     including ANSI, AVATAR and a subset of VT102.  This will accept
 *     nulls that are embedded within the string.
 */
void afprint(aterm_type *vid, unsigned char *astring, word len)
{
  int ptr = 0;                               /* Character pointer */
  int i,j;                                   /* Index/Temporary variables */
  byte l,c,x,y,x2,y2;                        /* Temporary variables */
  char *chr;
  static recur_level = 0;                    /* Recursion level counter */
  static recur_overflow = 0;                 /* Recursion overflow flag */

  /* The following loops for each character in the string to be output. */
  while(ptr < len) {

    /**************** PRINTABLE CHARS / BASIC CODES HANDLER ****************/
    if (vid->codetype == nul) {              /* If not processing codes */
      if (!ControlCode[astring[ptr]]) {        /* Fast shortcut for printing */
        i = ptr;
        while((ptr < len) && !ControlCode[astring[ptr]]) ptr++;
        outstring(vid,&astring[i],ptr-i);      /* Display string */
      }
      if (ptr < len) {
        switch (astring[ptr]) {
        case esc:     /* ANSI prefix */
          if (vid->ansi) {                     /* If ANSI enabled? */
            vid->codetype = esc;               /* Set ANSI code flag */
            vid->codeptr = 0;                  /* Set code buffer pointer */
          }
          else
            outstring(vid,&astring[ptr],1);    /* Display character */
          break;

        case avt:     /* AVATAR prefix */
          if (vid->avatar) {                   /* If AVATAR enabled? */
            vid->codetype = avt;               /* Set AVATAR code flag */
            vid->codeptr = 0;                  /* Set code buffer pointer */
          }
          else
            outstring(vid,&astring[ptr],1);    /* Display character */
          break;

        case rle:     /* RLE prefix */
          if (vid->avatar) {                   /* If AVATAR enabled? */
            vid->codetype = rle;               /* Set RLE code flag */
            vid->codeptr = 0;                  /* Set code buffer pointer */
          }
          else
            outstring(vid,&astring[ptr],1);    /* Display character */
          break;

        case 13:      /* Carriage Return */
          vid->col = 1;                        /* New cursor position */
          break;

        case 10:      /* Linefeed */
        case 11:      /* This also seems to do LF in many terminals */
          /* Move cursor down only if not at bottom of screen or scrollzone */
          if ((vid->row != vid->height) && (vid->row != vid->botzone))
            vid->row++;
          /* Scroll screen/scrollzone if at bottom of current scrollzone */
          else if (vid->row == vid->botzone)
            scrollup(vid,vid->wtopzone,vid->wleft,vid->wbotzone,vid->wright,1);
          break;

        case 8:       /* Backspace */
          if (vid->col > 1) {                  /* If not already leftmost */
            vid->col--;                        /* Cursor left */
            if (vid->destbs) pokechar(vid,' '); /* Destructive backspace? */
          }
          break;

        case 9:       /* Tab */
          vid->col = vid->col - ((vid->col - 1) % 8) + 8; /* Next tab stop */
          if (vid->col > vid->width)           /* Did it go off screen? */
            vid->col = vid->width;             /* Put cursor at right */
          break;

        case 12:      /* Clear screen */
          if (vid->ansi || vid->avatar) {      /* Must be ANSI or AVT mode */
            if (vid->avatar) vid->attr = 3;    /* Set cyan only in AVT mode */
            vid->row = 1;                      /* Home cursor */
            vid->col = 1;
            clearbox(vid, vid->wtop, vid->wleft, vid->wbot, vid->wright);
            break;
          }

        case 7:       /* Bell */
          if (vid->beep) beep();               /* Beep if bell allowed */
          break;

        default:      /* Printable character */
          outstring(vid,&astring[ptr],1);      /* Display character */
        }
      }
    } /* End of printable char/basic control code handling */


    /************************** ANSI/VT102 HANDLER *************************/
    else if (vid->codetype == esc) {         /* ANSI/VT102 code ^[ */

      vid->code[vid->codeptr] = astring[ptr];
      if (vid->codeptr == 0) {
        if (astring[ptr] == bracket) {
          vid->codeptr++;
        }

        else {
          /* The following code processes two-character escape sequences. */
          /* For those sequences with just an esc followed by a letter.   */
          vid->codetype = nul;

          switch (astring[ptr]) {
          case 'E':       /* Execute CRLF */
            vid->col = 1;
            /* Now execute statements below for LF */

          case 'D':       /* Execute LF */
            /* Move cursor down if not at bottom of screen or scrollzone */
            if ((vid->row != vid->height) && (vid->row != vid->botzone))
              vid->row++;
            /* Scroll screen/scrollzone if at bottom of scrollzone */
            else if (vid->row == vid->botzone)
              scrollup(vid,vid->wtopzone,vid->wleft,vid->wbotzone,vid->wright,1);
            break;

          case 'M':       /* Execute reverse LF */
            /* Move cursor up if not at top of screen or scrollzone */
            if ((vid->row != 1) && (vid->row != vid->topzone))
              vid->row--;
            /* Scroll screen/scrollzone if at top of scrollzone */
            else if (vid->row == vid->topzone)
              scrolldn(vid,vid->wtopzone,vid->wleft,vid->wbotzone,vid->wright,1);
            break;
          }
        }
      } /* if (vid->codeptr == 0) */

      else {

        /* If the code buffer is not already full, add char to code buffer */
        if (vid->codeptr < MaxCodeLen)
          vid->code[vid->codeptr++] = astring[ptr];

        /* If character is a suffix, then execute sequence in code buffer */
        if (ANSIchar[astring[ptr]] == SUFFIX) {
          vid->code[vid->codeptr] = nul;
          switch (astring[ptr]) {
          case 'm': {     /* Set attributes */
            chr = vid->code;
            do {
              c = strtol(++chr,NULL,10);
              chr = strchr(chr,';');
              switch (c) {
              case 30: case 31: case 32: case 33:
              case 34: case 35: case 36: case 37:  /* Foreground */
                vid->attr = (vid->attr & ~7) | (ANSI2IBM[c - 30]);
                break;
              case 40: case 41: case 42: case 43:
              case 44: case 45: case 46: case 47:  /* Background */
                vid->attr = (vid->attr & ~112) | (ANSI2IBM[c - 40] << 4);
                break;
              case 0:  /* Reset */
                vid->attr = 7;
                break;
              case 1:  /* Bold */
                vid->attr |= 8;
                break;
              case 5:  /* Blink */
                vid->attr |= 128;
                break;
              case 7:  /* Reverse */
                vid->attr = (vid->attr & 136) | ((vid->attr & 7) << 4) |
                            ((vid->attr & 112) >> 4);
                break;
              case 8:  /* Concealed */
                vid->attr = (vid->attr & 112) | ((vid->attr & 112) >> 4);
                break;
              }
            } while (chr != NULL);
            break;
          }

          case 'H':       /* Cursor locate */
          case 'f':       /* Cursor locate */
            y = strtol(&vid->code[1],NULL,10);
            if (y < 1) y = 1; else if (y > vid->height) y = vid->height;
            chr = strchr(vid->code,';');
            if (chr == NULL)
              x = 1;
            else {
              x = strtol(++chr,NULL,10);
              if (x < 1) x = 1; else if (x > vid->width) x = vid->width;
            }
            vid->row = y;
            vid->col = x;
            break;

          case 'A':       /* Cursor up */
            c = strtol(&vid->code[1],NULL,10);
            if (c < 1) c = 1;
            if ((vid->row >= vid->topzone) && ((vid->row - c) < vid->topzone))
              vid->row = vid->topzone;
            else if ((vid->row - c) < 1)
              vid->row = 1;
            else
              vid->row = vid->row - c;
            break;

          case 'B':       /* Cursor down */
            c = strtol(&vid->code[1],NULL,10);
            if (c < 1) c = 1;
            if ((vid->row <= vid->botzone) && ((vid->row + c) > vid->botzone))
              vid->row = vid->botzone;
            else if ((vid->row + c) > vid->height)
              vid->row = vid->height;
            else
              vid->row = vid->row + c;
            break;

          case 'C':       /* Cursor right */
            c = strtol(&vid->code[1],NULL,10);
            if (c <= 1) c = 1;
            if ((vid->col + c) > vid->width)
              vid->col = vid->width;
            else
              vid->col = vid->col + c;
            break;

          case 'D':       /* Cursor left */
            c = strtol(&vid->code[1],NULL,10);
            if (c < 1) c = 1;
            if ((vid->col - c) < 1)
              vid->col = 1;
            else
              vid->col = vid->col - c;
            break;

          case 'K':       /* Clear line */
            c = strtol(&vid->code[1],NULL,10);
            l = vid->wtop + vid->row - 1;
            if (c == 0)
              clearbox(vid, l, vid->wleft + vid->col - 1, l, vid->wright);
            else if (c == 1)
              clearbox(vid, l, vid->wleft, l, vid->wleft + vid->col - 1);
            else if (c == 2)
              clearbox(vid, l, vid->wleft, l, vid->wright);
            break;

          case 'J':       /* Clear screen */
            c = strtol(&vid->code[1],NULL,10);
            l = vid->wtop + vid->row - 1;
            if (c == 0) {
              clearbox(vid, l, vid->wleft + vid->col - 1, l, vid->wright);
              if (vid->row < vid->height)
                clearbox(vid, l+1, vid->wleft, vid->wbot, vid->wright);
            }
            else if (c == 1)  {
              clearbox(vid, l, vid->wleft, l, vid->wleft + vid->col - 1);
              if (vid->row > 1)
                clearbox(vid, vid->wtop, vid->wleft, l-1, vid->wright);
            }
            else if (c == 2) {
              vid->row = 1;
              vid->col = 1;
              clearbox(vid, vid->wtop, vid->wleft, vid->wbot, vid->wright);
            }
            break;

          case 's':       /* Save cursor position */
            vid->saverow = vid->row;
            vid->savecol = vid->col;
            break;

          case 'u':       /* Restore cursor position */
            vid->row = vid->saverow;
            vid->col = vid->savecol;
            break;

          case 'P':       /* Delete characters */
            c = strtol(&vid->code[1],NULL,10);
            if (c == 0) c = 1;
            l = vid->wtop + vid->row - 1;
            scrollleft(vid, l, vid->wleft + vid->col - 1, vid->wright, c);
            break;

          case '@':       /* Insert spaces */
            c = strtol(&vid->code[1],NULL,10);
            if (c == 0) c = 1;
            l = vid->wtop + vid->row - 1;
            scrollright(vid, l, vid->wleft + vid->col - 1, vid->wright, c);
            break;

          case 'M':       /* Delete lines */
            c = strtol(&vid->code[1],NULL,10);
            if (c == 0) c = 1;
            l = vid->wtop + vid->row - 1;
            scrollup(vid, l, vid->wleft, vid->wbot, vid->wright, c);
            break;

          case 'L':       /* Insert lines */
            c = strtol(&vid->code[1],NULL,10);
            if (c == 0) c = 1;
            l = vid->wtop + vid->row - 1;
            scrolldn(vid, l, vid->wleft, vid->wbot, vid->wright, c);
            break;

          case 'r':       /* Set scroll zone */
            y = strtol(&vid->code[1],NULL,10);
            chr = strchr(vid->code,';');
            if (chr == NULL)
              y2 = vid->height;
            else
              y2 = strtol(++chr,NULL,10);
            if (y > vid->height) {
              y = 1;
              y2 = vid->height;
            }
            else {
              if (y == 0) y = 1;
              if ((y > y2) || (y2 > vid->height)) y2 = vid->height;
            }
            vid->row = 1;
            vid->col = 1;
            vid->topzone = y;
            vid->botzone = y2;
            vid->wtopzone = vid->wtop + y - 1;
            vid->wbotzone = vid->wtop + y2 - 1;
            break;

          case 'n':       /* Status report */
            break;
          }
          vid->codetype = nul;
        }
      }
    }

    /**************************** AVATAR HANDLER ***************************/
    else if (vid->codetype == rle) {         /* RLE expansion ^Y */
      if (vid->codeptr == 0) {
      /* If it is beginning of buffer, add char to buffer (length of RLE) */
        vid->code[0] = astring[ptr];           /* Save char */
        vid->codeptr++;                        /* Inc buffer pointer */
      }
      else {
      /* If char already in buffer, then execute RLE command */
        j = astring[ptr];                      /* Get length of RLE */
        memset(vid->code,vid->code[0],j);      /* Create redundant string */
        outstring(vid,vid->code,j);            /* Output RLE string */
        vid->codetype = nul;                   /* End of RLE command */
      }
    }

    else if (vid->codetype == avt) {         /* AVATAR codes ^V */
      vid->code[vid->codeptr++] = astring[ptr];  /* Save char and inc ptr */

      if (vid->codeptr == 1) {                   /* First char after ^V */
        vid->codelen = AVTtable[astring[ptr]];   /* Length of AVT code */
        if (vid->codelen == 0)                   /* Invalid AVT code */
          vid->codetype = nul;
      }
      else if (vid->codelen == 255) {            /* For AVT code ^V^Y */
        vid->codelen = astring[ptr] + 3;         /* Actual leng of AVT code */
      }

      if (vid->codeptr >= vid->codelen) {
      /* Accumulate AVT string or execute it */

        /* If the AVT code is not ^V^Y then turn off Insert Mode */
        if (vid->code[0] != 25)
          vid->insmode = FALSE;

        switch (vid->code[0]) {
        case 1:                        /* ^V^A<A> set attributes */
          vid->attr = vid->code[1];
          break;

        case 2:                        /* ^V^B set blink on */
          vid->attr = vid->attr | 0x80;
          break;

        case 3:                        /* ^V^C move cursor up */
          if (vid->row > 1) vid->row--;
          break;

        case 4:                        /* ^V^D move cursor down */
          if (vid->row < vid->height) vid->row++;
          break;

        case 5:                        /* ^V^E move cursor left */
          if (vid->col > 1) vid->col--;
          break;

        case 6:                        /* ^V^F move cursor right */
          if (vid->col < vid->width) vid->col++;
          break;

        case 7:                        /* ^V^G Clear to end of line */
          clearbox(vid,vid->wtop + vid->row - 1, vid->wleft + vid->col - 1,
                       vid->wtop + vid->row - 1, vid->wright);
          break;

        case 8:                        /* ^V^H<Y><X> Locate cursor */
          vid->row = vid->code[1];
          vid->col = vid->code[2];
          if (vid->row == 0)
            vid->row = 1;
          else if (vid->row > vid->height)
            vid->row = vid->height;
          if (vid->col == 0)
            vid->col = 1;
          else if (vid->col > vid->width)
            vid->col = vid->width;
          break;

        case 9:                        /* ^V^I Set insert mode on */
          vid->insmode = TRUE;
          break;

        case 10:                       /* ^V^J<L,Y,X,Y2,X2> Scroll block up */
        case 11:                       /* ^V^K<L,Y,X,Y2,X2> Scroll block dn */
          l = vid->code[1];
          y = vid->code[2];
          x = vid->code[3];
          y2 = vid->code[4];
          x2 = vid->code[5];
          if (y  == 0) y  = 1; else if (y  > vid->height) y  = vid->height;
          if (x  == 0) x  = 1; else if (x  > vid->width)  x  = vid->width;
          if ((y > y2) || (x > x2)) break;
          if (y2 > vid->height) y2 = vid->height;
          if (x2 > vid->width)  x2 = vid->width;
          if (vid->code[0] == 10) {
            scrollup(vid, vid->wtop + y - 1,  vid->wleft + x - 1,
                          vid->wtop + y2 - 1, vid->wleft + x2 - 1, l);
          }
          else {
            scrolldn(vid, vid->wtop + y - 1,  vid->wleft + x - 1,
                          vid->wtop + y2 - 1, vid->wleft + x2 - 1, l);
          }
          break;

        case 12:                       /* ^V^L<A,Y,X> Clear block */
          vid->attr = vid->code[1];
          y = vid->code[2];
          x = vid->code[3];
          if ((x <= 0) || (y <= 0)) break;
          y = y + vid->wtop + vid->row - 2;
          x = x + vid->wleft + vid->col - 2;
          if (y > vid->wbot) y = vid->wbot;
          if (x > vid->wright) x = vid->wright;
          clearbox(vid, vid->wtop + vid->row - 1,
                   vid->wleft + vid->col - 1, y, x);
          break;

        case 13:                       /* ^V^M<A,C,Y,X> Fill block */
          vid->attr = vid->code[1];
          c = vid->code[2];
          y = vid->code[3];
          x = vid->code[4];
          if ((x <= 0) || (y <= 0)) break;
          y = y + vid->wtop + vid->row - 2;
          x = x + vid->wleft + vid->col - 2;
          if (y > vid->wbot) y = vid->wbot;
          if (x > vid->wright) x = vid->wright;
          fillbox(vid, vid->wtop + vid->row - 1,
                       vid->wleft + vid->col - 1, y, x, c);
          break;

        case 14:                       /* ^V^N Delete character */
          scrollleft(vid, vid->wtop + vid->row - 1, vid->wleft + vid->col - 1,
                          vid->wright, 1);
          break;

        case 25:                       /* ^V^Y Repeat String */
          /* Were there too many levels of recursive calls to afprint? */
          if (recur_level >= MaxRecurseLevels)
            recur_overflow = TRUE;       /* Start unravelling recursion */

          /* Allow recursive call if not in recursion overflow state */
          else if (!recur_overflow) {
            /* Save the current code buffer and pointers */
            byte temp_codetype = vid->codetype;
            byte temp_codelen = vid->codelen;
            word temp_codeptr = vid->codeptr;
            byte temp_code[MaxCodeLen];
            memcpy(temp_code,vid->code,MaxCodeLen);

            /* Execute recursive call to afprint function */
            recur_level++;              /* Protection */
            vid->codetype = 0;
            for (i = 1; i <= temp_code[temp_codelen-1]; i++) {
              afprint(vid,&temp_code[2],temp_code[1]);
            }
            recur_level--;              /* Protection */

            /* Reset recursion overflow flag if no more levels of recursion */
            if (!recur_level) recur_overflow = FALSE;

            /* Restore code buffer and pointers in order to resume */
            /* processing the rest of the code buffer */
            vid->codetype = temp_codetype;
            vid->codelen = temp_codelen;
            vid->codeptr = temp_codeptr;
            memcpy(vid->code,temp_code,MaxCodeLen);
          }
          break;
        }
        vid->codetype = nul;           /* End of AVT code */
      }
    }
    /*********************** END OF CODE HANDLERS **************************/

    ptr++;                                   /* Increment character pointer */
  } /* end of while loop*

  /* Update actual cursor position */
  poscur(vid,vid->row,vid->col);
}
