/* atype.c
 *    ANSI/AVATAR/VT102 display command, version 2.0
 *    Based on the original QuickBASIC-based ATYPE 1.03
 *    Programmed 1992, Feb 1995 by Mark D. Rejhon.
 *    EMail: marky@ottawa.com, ag115@freenet.carleton.ca
 *
 * WHAT IS IT?
 *    This is a MSDOS 'type' command substitute that has these advantages:
 *    - No ANSI.SYS required.
 *    - AVATAR/0+ and VT102 processing.
 *    - Speed adjustable in characters per second at the command line!
 *
 * FILES
 *    atype.c   - This main code module.
 *    aterm.h   - The main include module for the display functions.
 *    adefs.h   - The definitions and constants module for aterm.h.
 *    avid_tc.h - Turbo C++ specific video display functions.
 *
 * LICENSE:
 *    This version can be freely modified and redistributed using the GNU
 *    public license.  Please refer to included "COPYING" file for details.
 *    Please submit ALL changes to this code via Internet Email to address
 *    ag115@freenet.carleton.ca (permanent address, will forward Email to
 *    my current Internet Service Provider)
 */

#include "aterm.h"

/* Maximum block size when reading the ansi/avt file */
#define MaxSize 1024

/* This is the Ctrl-Break handler */
int c_break(void)
{
   return(0);
}

int main(int argc, char *argv[])
{
  /* Internal variable declarations */
  unsigned char *bigbuffer[MaxSize+4];
  FILE *stream;
  int non_slashed, help;          /* Flags for command line parms */
  int rows_parm, cols_parm;       /* Command line rows/cols */
  int rows, cols;                 /* Detected rows/cols */
  int rows_set, cols_set;         /* Actual rows/cols for text display */
  int i, height;
  unsigned long bsize, delay_len; /* For adjusting display speed */

  /* Parameter Variables */
  char *filename;
  unsigned long speed;

  /* Terminal window declarations */
  aterm_type video;
  aterm_type *vid = &video;

  /* Set Ctrl-Break handler */
  ctrlbrk(c_break);

  rows_parm = 0;
  cols_parm = 0;
  speed = 0;
  non_slashed = 0;
  help = 0;
  for (i = 1; i < argc; i++) {
    if (argv[i][0] == '/') {
      switch (argv[i][1]) {
      case 'H':                  /* Set scrolling height */
        rows_parm = atoi(&argv[i][2]);
        break;
      case 'W':                  /* Set wrapping width */
        cols_parm = atoi(&argv[i][2]);
        break;
      default:                   /* Command line help */
        help = 1;
        break;
      }
    }
    else {
      switch (non_slashed) {
      case 0: filename = argv[i];    break;
      case 1: speed = atol(argv[i]); break;
      }
      non_slashed++;
    }
  }

  /* Display help if help switch was used, or no filename was specified */
  if (!non_slashed || help) {
    printf("\n"
    "ANSI/AVATAR file displaying utility version 2.0\n"
    "Supports AVATAR/0+ and most ANSI/VT102 codes!\n"
    "Programmed 1992, Feb 1995 by Mark D. Rejhon.\n"
    "\n"
    "  ATYPE <filename> [speed] [/Hnn] [/?]\n"
    "\n"
    "  filename....Filename of file with ANSI/AVATAR codes.\n"
    "  speed.......An optional number specifying speed in characters per second.\n"
    "  /Hnn........Optional switch with number nn up to current screen height,\n"
    "              to set height of scrolling region to emulate a terminal with a\n"
    "              specific number of lines.  Screen height is used by default.\n"
    "  /Wnn........Optional switch with number nn up to current screen width,\n"
    "              to set column of text wrapping, to emulate a terminal with a\n"
    "              narrower screen.  Screen width is used by default.\n"
    );
    return 3;
  }

  /* Figure out screen size by checking BIOS data area */
  cols = peekb(0x0040,0x004A);
  rows = peekb(0x0040,0x0084)+1;

  /* Use command line rows value if it exists and is within bounds */
  if ((rows_parm > 0) && (rows_parm < rows)) {
    /* Scroll the screen so the unused lines become blank */
    height = (rows - rows_parm);
    for (i = 1; i <= height; i++) printf("\n");
    gotoxy(1, wherey() - height);
    rows_set = rows_parm;
  }
  else
    rows_set = rows;

  /* Use command line columns value if it exists and is within bounds */
  if ((cols_parm > 0) && (cols_parm < cols))
    cols_set = cols_parm;
  else
    cols_set = cols;

  /* Initialize terminal screen, and set cursor position */
  ainit(vid,1,1,rows_set,cols_set);
  vid->row = wherey();
  vid->scrwidth = cols;

  if (speed) {
    /* Compute the string block size based on speed value, adjusted so that */
    /* a max 70 blocks are printed per second, for efficiency purposes. */
    bsize = speed / 70;
    if (bsize < 1) bsize = 1;
    if (bsize > MaxSize) bsize = MaxSize;

    /* Compute the delay in milliseconds between blocks */
    delay_len = 1000 * bsize / speed;
  }
  else {
    /* Speed was given as 0 so use maximum possible speed */
    bsize = MaxSize;
    delay_len = 0;
  }

  /* Open the file stream */
  stream = fopen(filename, "rb");
  if (stream == NULL) {                 /* Error opening file */
    printf("ERROR: The specified file does not exist, or could not be accessed.\n"
           "Please type 'ATYPE' alone at the command prompt for help.\n");
    return(1);
  }

  /* Start at the beginning of stream */
  rewind(stream);

  /* Loop to display file */
  while (!feof(stream)) {
    /* Read data from file and print it */
    i = fread(bigbuffer,1,bsize,stream);
    bigbuffer[i] = nul;
    afprint(vid, (byte*) bigbuffer,i);

    /* Pausing loop */
    if (delay_len) delay(delay_len);

    /* Process the keyboard if needed */
    if (kbhit()) {
      i = getch();
      if ((i == 27)||(i == 'S')||(i == 's')||(i == 3)) {       /* Abort */
        fclose(stream);
        exit(2);
      }
      if (i == 32) {                                           /* Pause */
        while (!kbhit())
          ;
        i = getch();
      }
    }
  }

  /* Close file stream */
  fclose(stream);
  return(0);
}
