/* avid_tc.h
 *    Include module for "aterm.h" for display functions to use with Turbo C++
 *    This module is used by "aterm.h".  See "avid_nul.h" for more info.
 *    Programmed 1994, Feb 1995 by Mark D. Rejhon.
 *    EMail: marky@ottawa.com, ag115@freenet.carleton.ca
 *
 * LICENSE:
 *    This version can be freely modified and redistributed using the GNU
 *    public license.  Please refer to included "COPYING" file for details.
 *    Please submit ALL changes to this code via Internet Email to address
 *    ag115@freenet.carleton.ca  (permanent address, will forward Email to
 *    my current Internet Service Provider)
 *
 * --------------------------------------------------------------------------
 * NOTES ABOUT FUNCTIONS
 *    - All these return void, all values are int unless otherwise specified.
 *    - All co-ordinate values should have an origin of (1,1).
 *    - Error checking on bad co-ordinate values are not critical if used
 *       mainly with the aprint/afprint terminal display functions.
 *    - See adefs.h for information on the 'video variables structure'.
 *
 * SUMMARY
 *    beep()
 *    poscur(aterm_type *vid, row, col)
 *    clearbox(aterm_type *vid, top, left, bot, right)
 *    fillbox(aterm_type *vid, top, left, bot, right, chr)
 *    scrollup(aterm_type *vid, top, left, bot, right, lines)
 *    scrolldn(aterm_type *vid, top, left, bot, right, lines)
 *    scrollleft(aterm_type *vid, linenum, left, right, chars)
 *    scrollright(aterm_type *vid, linenum, left, right, chars)
 *    pokechar(aterm_type *vid, char outch)
 *    outstring(aterm_type *vid, char *outstr, len)
 */

/* Make sure we define everything only once */
#ifndef AVID_H
#define AVID_H

/* Note: To use DOS.H, use Turbo C++ mode in Options|Compiler|Source */
#include <dos.h>
#include <conio.h>

/* Include adefs.h in case it was not included yet */
#include "adefs.h"


/* Produces a brief beep sound on the PC speaker */
void beep()
{
  sound(500);
  delay(25);
  nosound();
}


/* Position cursor on the screen */
void poscur(aterm_type *vid, int row, int col)
{
  static union REGS regs;            /* 8086 registers */
  regs.h.ah = 0x2;                   /* Cursor Position */
  regs.h.bh = 0;                     /* Video page 0 */
  regs.h.dh = vid->wtop + row - 2;   /* Cursor row */
  regs.h.dl = vid->wleft + col - 2;  /* Cursor column */
  int86(0x10, &regs, &regs);         /* Call VIDEO BIOS */
}


/* Clears a block of text on the screen */
void clearbox(aterm_type *vid, int top, int left, int bot, int right)
{
  static union REGS regs;       /* 8086 registers */
  regs.h.ah = 0x6;              /* Scroll Window Interrupt */
  regs.h.al = 0;                /* Blank entire window */
  regs.h.bh = vid->attr;        /* Attributes to use */
  regs.h.ch = top - 1;          /* Row of upper left corner */
  regs.h.cl = left - 1;         /* Col of upper left corner */
  regs.h.dh = bot - 1;          /* Row of lower right corner */
  regs.h.dl = right - 1;        /* Col of lower right corner */
  int86(0x10, &regs, &regs);    /* Call VIDEO BIOS */
}


/* Fills block of text on screen, to specific char, and attrs in vid->attr */
void fillbox(aterm_type *vid, int top, int left, int bot, int right, int chr)
{
  static int i,j;
  static word tmp, value;
  value = chr + (vid->attr << 8);
  for (i = top; i <= bot; i++) {
    tmp = ((i - 1) * vid->scrwidth + (left - 1)) << 1;
    for (j = left; j <= right; j++) {
      poke(ScreenSeg,tmp,value);
      tmp += 2;
    }
  }
}


/* Scrolls a block of text on the screen upwards. */
void scrollup(aterm_type *vid, int top, int left, int bot, int right,
              int lines)
{
  if ((lines > 0) && (lines <= bot-top)) {
    static union REGS regs;       /* 8086 registers */
    regs.h.ah = 0x6;              /* Scroll Window Interrupt */
    regs.h.al = lines;            /* Blank entire window */
    regs.h.bh = vid->attr;        /* Attributes to use for gap */
    regs.h.ch = top - 1;          /* Row of upper left corner */
    regs.h.cl = left - 1;         /* Col of upper left corner */
    regs.h.dh = bot - 1;          /* Row of lower right corner */
    regs.h.dl = right - 1;        /* Col of lower right corner */
    int86(0x10, &regs, &regs);    /* Call VIDEO BIOS */
  }
  else {
    clearbox(vid, top, left, bot, right);
  }
}


/* Scrolls a block of text on the screen downwards. */
void scrolldn(aterm_type *vid, int top, int left, int bot, int right,
              int lines)
{
  if ((lines > 0) && (lines <= bot-top)) {
    static union REGS regs;       /* 8086 registers */
    regs.h.ah = 0x7;              /* Scroll Window Interrupt */
    regs.h.al = lines;            /* Blank entire window */
    regs.h.bh = vid->attr;        /* Attributes to use for gap */
    regs.h.ch = top - 1;          /* Row of upper left corner */
    regs.h.cl = left - 1;         /* Col of upper left corner */
    regs.h.dh = bot - 1;          /* Row of lower right corner */
    regs.h.dl = right - 1;        /* Col of lower right corner */
    int86(0x10, &regs, &regs);    /* Call VIDEO BIOS */
  }
  else {
    clearbox(vid, top, left, bot, right);
  }
}


/* Scrolls part of a line of text on the screen towards the left. */
void scrollleft(aterm_type *vid, int linenum, int left, int right, int chars)
{
  if ((right-left >= chars) && chars > 0) {
    movetext(left+chars, linenum, right, linenum, left, linenum);
    clearbox(vid, linenum, right-chars+1, linenum, right);
  }
  else {
    clearbox(vid, linenum, left, linenum, right);
  }
}


/* Scrolls part of a line of text on the screen towards the right. */
void scrollright(aterm_type *vid, int linenum, int left, int right, int chars)
{
  if ((right-left >= chars) && chars > 0) {
    movetext(left, linenum, right-chars, linenum, left+chars, linenum);
    clearbox(vid, linenum, left, linenum, left+chars-1);
  }
  else {
    clearbox(vid, linenum, left, linenum, right);
  }
}


/* Writes a char to screen without affecting cursor position. */
void pokechar(aterm_type *vid, char outch)
{
  static int tmp;
  tmp = ((vid->row + vid->wtop - 2) * vid->scrwidth + (vid->col + vid->wleft - 2)) << 1;
  pokeb(ScreenSeg,tmp++,outch);
  pokeb(ScreenSeg,tmp,vid->attr);
}


/* Output a string to screen, all control codes are printable */
void outstring(aterm_type *vid, char *outstr, int len)
{
  static int i,tmp;

  /* Insert mode, check if it's time to scroll text to the right */
  if (vid->insmode) {
    if (len <= (vid->width - vid->col))
      scrollright(vid, vid->wtop + vid->row - 1,
                  vid->wleft + vid->col - 1, vid->wright, len);
  }
  /* Screen memory location of current char position */
  tmp = ((vid->row + vid->wtop - 2) * vid->scrwidth + (vid->col + vid->wleft - 2)) << 1;

  i = 0;
  while(i < len) {
    /* Write char and attr to screen */
    pokeb(ScreenSeg,tmp++,outstr[i++]);
    pokeb(ScreenSeg,tmp++,vid->attr);

    /* Is it end of a screen row? */
    if (vid->col >= vid->width) {
      /* Set cursor variables to start of next row */
      vid->col = 1;
      if (vid->row == vid->botzone)
        scrollup(vid,vid->wtopzone,vid->wleft,vid->wbotzone,vid->wright,1);
      else if (vid->row != vid->height)
        vid->row++;

      /* Insert mode, check if it's time to scroll text to the right */
      if (vid->insmode) {
        if (len <= (vid->width - vid->col))
          scrollright(vid, vid->wtop + vid->row - 1,
                           vid->wleft + vid->col - 1, vid->wright, len-i);
      }

      /* Screen memory location of start of next row */
      tmp = ((vid->row + vid->wtop - 2) * vid->scrwidth + (vid->col + vid->wleft - 2)) << 1;
    }
    else
      vid->col++;             /* Increment cursor col */
  }
}

#endif
