/* avid_mem.h
 *    Include module for "aterm.h" for memory buffer video display functions
 *    This module is used by "aterm.h".  See "avid_nul.h" for more info.
 *    Programmed 1994, Feb 1995 by Mark D. Rejhon.
 *    EMail: marky@ottawa.com, ag115@freenet.carleton.ca
 *
 * LICENSE:
 *    This version can be freely modified and redistributed using the GNU
 *    public license.  Please refer to included "COPYING" file for details.
 *    Please submit ALL changes to this code via Internet Email to address
 *    ag115@freenet.carleton.ca  (permanent address, will forward Email to
 *    my current Internet Service Provider)
 *
 * --------------------------------------------------------------------------
 * NOTES ABOUT FUNCTIONS
 *    - All these return void, all values are int unless otherwise specified.
 *    - All co-ordinate values should have an origin of (1,1).
 *    - Error checking on bad co-ordinate values are not critical if used
 *       mainly with the aprint/afprint terminal display functions.
 *    - See adefs.h for information on the 'video variables structure'.
 *
 * SUMMARY
 *    beep()
 *    poscur(aterm_type *vid, row, col)
 *    clearbox(aterm_type *vid, top, left, bot, right)
 *    fillbox(aterm_type *vid, top, left, bot, right, chr)
 *    scrollup(aterm_type *vid, top, left, bot, right, lines)
 *    scrolldn(aterm_type *vid, top, left, bot, right, lines)
 *    scrollleft(aterm_type *vid, linenum, left, right, chars)
 *    scrollright(aterm_type *vid, linenum, left, right, chars)
 *    pokechar(aterm_type *vid, char outch)
 *    outstring(aterm_type *vid, char *outstr, len)
 */

/* Make sure we define everything only once */
#ifndef AVID_H
#define AVID_H

/* Include adefs.h in case it was not included yet */
#include "adefs.h"


/* Produce a brief beep sound */
void beep()
{ /* Dummy function, not used for internal text buffer */ }


/* Position cursor on video buffer. */
void poscur(aterm_type *vid, int row, int col)
{ /* Dummy function, not used for internal text buffer */ }


/* Clears a block of text on the video buffer. */
#define clearbox(vid, top, left, bot, right) \
        fillbox(vid, top, left, bot, right, 32)


/* Fills a block of text on the video buffer to a specific character. */
void fillbox(aterm_type *vid, int top, int left, int bot, int right, int chr)
{
  static int i, j, tmp, value;

  /* Loop to fill each row, one row at a time */
  for (i = top; i <= bot; i++) {
    /* Calculate pointer to first character of row number i */
    tmp = ((i - 1) * vid->scrwidth + (left - 1)) << 1;

    /* Loop to fill each column one char at a time */
    for (j = left; j <= right; j++) {
      vid->buffer[tmp++] = chr;          /* Write character */
      vid->buffer[tmp++] = vid->attr;    /* Write attributes */
    }
  }
}


/* Scrolls a block of text on the video buffer upwards. */
void scrollup(aterm_type *vid, int top, int left, int bot, int right,
              int lines)
{
  /* Check No. of lines to be positive, and less than height of text box */
  if ((lines > 0) && (lines <= bot-top)) {
    static int tmp_src, tmp_dest, i;
    static size_t wid;

    /* Compute width of text box being scrolled */
    wid = (right - left + 1) * 2;

    /* Loop for scrolling text box, by copying one line at a time */
    for (i = top; i <= (bot - lines); i++) {
      /* Compute source and destination of first char of a line */
      tmp_src = ((i + lines - 1) * vid->scrwidth + (left - 1)) << 1;
      tmp_dest = ((i - 1) * vid->scrwidth + (left - 1)) << 1;

      /* Copy the line of text to its new line position */
      memcpy(&vid->buffer[tmp_dest],&vid->buffer[tmp_src],wid);
    }
    /* Fill the gap with blanks */
    clearbox(vid, bot - lines + 1, left, bot, right);
  }
  else {
    /* Fill entire text box with blanks */
    clearbox(vid, top, left, bot, right);
  }
}


/* Scrolls a block of text on the video buffer downwards. */
void scrolldn(aterm_type *vid, int top, int left, int bot, int right,
              int lines)
{
  /* Check No. of lines to be positive, and less than height of text box */
  if ((lines > 0) && (lines <= bot-top)) {
    static int tmp_src, tmp_dest, i;
    static size_t wid;

    /* Compute width of text box being scrolled */
    wid = (right - left + 1) * 2;

    /* Loop for scrolling text box, by copying one line at a time */
    for (i = (bot - lines); i >= top; i--) {
      /* Compute source and destination of first char of a line */
      tmp_dest = ((i + lines - 1) * vid->scrwidth + (left - 1)) << 1;
      tmp_src = ((i - 1) * vid->scrwidth + (left - 1)) << 1;

      /* Copy the line of text to its new line position */
      memcpy(&vid->buffer[tmp_dest],&vid->buffer[tmp_src],wid);
    }
    /* Fill the gap with blanks */
    clearbox(vid, top, left, top + lines - 1, right);
  }
  else {
    /* Fill entire text box with blanks */
    clearbox(vid, top, left, bot, right);
  }
}


/* Scrolls part of a line of text in the video buffer towards the left. */
void scrollleft(aterm_type *vid, int linenum, int left, int right, int chars)
{
  if ((right-left >= chars) && chars > 0) {
    static int tmp_src, tmp_dest;
    static size_t len;

    /* Length of text that that are actually moved, including attributes */
    len = (right - left - chars + 1) * 2;

    /* Compute source and destination of first character */
    tmp_dest = ((linenum - 1) * vid->scrwidth + (left - 1)) << 1;
    tmp_src = tmp_dest + (chars << 1);

    /* Execute the move of text */
    memmove(&vid->buffer[tmp_dest],&vid->buffer[tmp_src],len);

    /* Fill gap with blanks */
    clearbox(vid, linenum, right-chars+1, linenum, right);
  }
  else {
    /* Fill entire area with blanks */
    clearbox(vid, linenum, left, linenum, right);
  }
}


/* Scrolls part of a line of text in the video buffer towards the right. */
void scrollright(aterm_type *vid, int linenum, int left, int right, int chars)
{
  if ((right-left >= chars) && chars > 0) {
    static int tmp_src, tmp_dest;
    static size_t len;

    /* Length of text that that are actually moved, including attributes */
    len = (right - left - chars + 1) * 2;

    /* Compute source and destination of first character */
    tmp_src = ((linenum - 1) * vid->scrwidth + (left - 1)) << 1;
    tmp_dest = tmp_src + (chars << 1);

    /* Execute the move of text */
    memmove(&vid->buffer[tmp_dest],&vid->buffer[tmp_src],len);

    /* Fill gap with blanks */
    clearbox(vid, linenum, left, linenum, left+chars-1);
  }
  else {
    /* Fill entire area with blanks */
    clearbox(vid, linenum, left, linenum, right);
  }
}


/* Writes a char to video buffer without affecting cursor position. */
void pokechar(aterm_type *vid, char outch)
{
  static int tmp;

  /* Compute pointer into buffer array and write character */
  tmp = ((vid->row + vid->wtop - 2) * vid->scrwidth + (vid->col + vid->wleft - 2)) << 1;
  vid->buffer[tmp++] = outch;
  vid->buffer[tmp] = vid->attr;
}


/* Output a string to video buffer, all control codes are printable */
void outstring(aterm_type *vid, char *outstr, int len)
{
  static int i,tmp;

  /* Insert mode, check if it's time to scroll text to the right */
  if (vid->insmode) {
    if (len <= (vid->width - vid->col))
      scrollright(vid, vid->wtop + vid->row - 1,
                  vid->wleft + vid->col - 1, vid->wright, len);
  }
  /* Screen memory location of current char position */
  tmp = ((vid->row + vid->wtop - 2) * vid->scrwidth + (vid->col + vid->wleft - 2)) << 1;

  i = 0;
  while(i < len) {
    /* Write char and attr to screen buffer */
    vid->buffer[tmp++] = outstr[i++];
    vid->buffer[tmp++] = vid->attr;

    /* Is it end of a screen row? */
    if (vid->col >= vid->width) {
      /* Set cursor variables to start of next row */
      vid->col = 1;
      if (vid->row == vid->botzone)
        scrollup(vid,vid->wtopzone,vid->wleft,vid->wbotzone,vid->wright,1);
      else if (vid->row != vid->height)
        vid->row++;

      /* Insert mode, check if it's time to scroll text to the right */
      if (vid->insmode) {
        if (len <= (vid->width - vid->col))
          scrollright(vid, vid->wtop + vid->row - 1,
                           vid->wleft + vid->col - 1, vid->wright, len-i);
      }

      /* Screen memory location of start of next row */
      tmp = ((vid->row + vid->wtop - 2) * vid->scrwidth + (vid->col + vid->wleft - 2)) << 1;
    }
    else
      vid->col++;             /* Increment cursor col */
  }
}

#endif
